import http from "k6/http";
import { check, fail, sleep } from "k6";
import { SharedArray } from "k6/data";
import papaparse from "https://jslib.k6.io/papaparse/5.1.1/index.js";
import { textSummary } from "https://jslib.k6.io/k6-summary/0.0.1/index.js";

// ============================
// 1. LOAD CSV USER DATA (minimal 50 user login)
// Format CSV: username,password
// ============================
const users = new SharedArray("User data", function () {
  return papaparse
    .parse(open("./data/users.csv"), { header: true })
    .data.slice(0, 50);
});

// ============================
// 2. CONFIG: Lonjakan cepat ke 50 VU, lalu tahan 1 menit
// ============================
export let options = {
  stages: [
    { duration: "10s", target: 50 }, // lonjakan cepat 5 → 50
    { duration: "1m", target: 50 }, // tahan 50 VU
    { duration: "5s", target: 0 }, // ramp-down cepat
  ],
  thresholds: {
    http_req_failed: ["rate<0.1"], // error rate < 10%
    http_req_duration: ["p(95)<4000"], // 95% response < 4s
  },
};

// ============================
// 3. MAIN LOGIN FUNCTION
// ============================
export default function () {
  if (__VU > users.length) {
    console.warn(`❌ VU ${__VU} dilewati karena user data tidak tersedia.`);
    return;
  }

  const user = users[__VU - 1];

  const payload = JSON.stringify({
    username: user.username.trim(),
    password: user.password.trim(),
  });

  const headers = { "Content-Type": "application/json" };
  const res = http.post("https://duwijayaunderstel.cloud/api/login", payload, {
    headers,
  });

  const success = check(res, {
    "✅ Login status 200": (r) => r.status === 200,
    "✅ Ada token": (r) => r.body.includes("token"),
  });

  if (!success) {
    console.error(`❌ Login gagal | Status: ${res.status} | Body: ${res.body}`);
    fail("❌ Gagal login (error resilience)");
  }

  sleep(1); // simulasi user delay
}

// ============================
// 4. SUMMARY OUTPUT
// ============================
export function handleSummary(data) {
  return {
    "resilience_summary.txt": textSummary(data, {
      indent: " ",
      enableColors: false,
    }),
    "resilience_summary.json": JSON.stringify(data, null, 2),
  };
}
