import http from "k6/http";
import { check, fail, sleep } from "k6";
import { textSummary } from "https://jslib.k6.io/k6-summary/0.0.1/index.js";

// === 1. Konfigurasi Test ===
export let options = {
  vus: 20,
  duration: "1m",
  thresholds: {
    http_req_failed: ["rate<0.05"],
    http_req_duration: ["p(95)<3000"],
  },
};

// === 2. Login Sekali di Setup ===
export function setup() {
  const loginRes = http.post(
    "https://ggbpuhwbr1.execute-api.us-east-1.amazonaws.com/dev/api/login",
    JSON.stringify({
      username: "q",
      password: "q",
    }),
    {
      headers: { "Content-Type": "application/json" },
    }
  );

  const ok = check(loginRes, {
    "✅ Login berhasil (setup)": (r) =>
      r.status === 200 && r.body.includes("token"),
  });

  if (!ok) {
    fail(`❌ Login gagal di setup | Status: ${loginRes.status}`);
  }

  const token = JSON.parse(loginRes.body).token;
  return { token }; // dikirim ke semua VU
}

// === 3. Fungsi Utama: Pakai Token dari setup ===
export default function (data) {
  const token = data.token;
  const headers = {
    "Content-Type": "application/json",
    Authorization: `Bearer ${token}`,
  };

  const itemData = {
    item: `Oli_${__VU}_${__ITER}_${Math.floor(Math.random() * 99999)}`,
    stock: Math.floor(Math.random() * 10) + 1,
    distributor_price: 50000,
    selling_price: 75000,
  };

  const res = http.post(
    "https://ggbpuhwbr1.execute-api.us-east-1.amazonaws.com/dev/api/item",
    JSON.stringify(itemData),
    {
      headers,
    }
  );

  const success = check(res, {
    "✅ Tambah item sukses": (r) => r.status === 201,
  });

  if (!success) {
    console.error(
      `❌ Gagal tambah item | Status: ${res.status} | Body: ${res.body}`
    );
    fail("Tambah item gagal");
  }

  sleep(1); // simulasi delay antar transaksi
}

// === 4. Ringkasan Output ===
export function handleSummary(data) {
  return {
    "tc-k02_tambah_item_summary.txt": textSummary(data, {
      indent: " ",
      enableColors: false,
    }),
    "tc-k02_tambah_item_summary.json": JSON.stringify(data, null, 2),
  };
}
