import http from "k6/http";
import { check, sleep, fail } from "k6";
import { SharedArray } from "k6/data";
import papaparse from "https://jslib.k6.io/papaparse/5.1.1/index.js";
import { textSummary } from "https://jslib.k6.io/k6-summary/0.0.1/index.js";

// ========== 1. LOAD USER LOGIN DATA ==========
const users = new SharedArray("User Login Data", function () {
  return papaparse
    .parse(open("./data/users.csv"), { header: true })
    .data.filter((u) => u.username && u.password)
    .slice(0, 10); // hanya ambil 10 user
});

// ========== 2. KONFIGURASI NORMAL LOAD ==========
export let options = {
  vus: 10,
  duration: "2m",
  thresholds: {
    http_req_duration: ["p(95)<3000"], // 95% response time < 3 detik
    http_req_failed: ["rate<0.05"], // error rate < 5%
  },
};

// ========== 3. PEMBERSIH STRING ==========
function clean(str) {
  return str ? str.trim().replace(/\r/g, "") : "";
}

// ========== 4. MAIN LOGIN FUNCTION ==========
export default function () {
  if (__VU > users.length) {
    console.warn(`❌ VU ${__VU} dilewati karena user data tidak tersedia.`);
    return;
  }

  const user = users[__VU - 1];

  const loginPayload = JSON.stringify({
    username: clean(user.username),
    password: clean(user.password),
  });

  const headers = { "Content-Type": "application/json" };
  const res = http.post(
    "https://duwijayaunderstel.cloud//api/login",
    loginPayload,
    { headers }
  );

  const ok = check(res, {
    "✅ Login berhasil (200)": (r) => r.status === 200,
    "✅ Ada token di body": (r) => r.body.includes("token"),
  });

  if (!ok) {
    console.error(`❌ Login gagal | Status: ${res.status} | Body: ${res.body}`);
    fail("❌ Login gagal");
  }

  sleep(1); // simulasi user delay
}

// ========== 5. OUTPUT SUMMARY ==========
export function handleSummary(data) {
  return {
    "normal_login_summary.txt": textSummary(data, {
      indent: " ",
      enableColors: false,
    }),
    "normal_login_summary.json": JSON.stringify(data, null, 2),
  };
}
